//
//  Ninereeds Broadcast Lib
//
//  This is the header for 'Ninereeds Broadcast Lib.DLL', which is the 'server' for broadcast
//  commands sent by the 'Ninereeds Broadcast' machine.
//
//  The idea is that machines with an interest in receiving such commands register a callback,
//  and that every command sent over a particular channel causes a callback to every machine
//  that is listening to that channel.
//
//  The command format is simple and general. Listeners can interpret commands however they
//  want - the intention is that listeners have a dialog or property sheet page where they can
//  assign a meaning to particular commands, so the user would see something like...
//
//    Listening to Channel    1
//
//    Command                 1    <-  combo box or spin control
//
//    Filter Cutoff           Unchanged
//    Filter Resonance        Byte Param 1   <-  combo box selects action for filter resonance etc
//
//  The reasons for this are...
//
//    1.  To provide extensibility for machines that have used all their parameters.
//    2.  To allow global settings changes (the mood at that stage in a tune) to be set
//        in one machine, and have multiple machines react to them - perhaps in different ways.
//    3.  To allow the possibility of interactive music, where an external application has a
//        standard interface to make changes to a buzz tune, reflecting a particular location
//        in a game or whatever. An external app would simply load this DLL and send the commands
//        the same way a broadcast machine would.
//
//  There should only ever be a need for one broadcaster machine (though there may be multiple
//  instances for multiple channels), but listeners must be specifically written to use this
//  information.
//
//  Senders need not register - it is up to the user to either avoid multiple-senders on the same
//  channel or ensure they cause no problems.
//
//  Channel numbers are not bounded - any +ve int will do. If a sender sends to a channel with no
//  listeners then that command is simply ignored. However, there will be an upper limit on the
//  total number of listeners that can register, and registering a new listener can therefore
//  fail.
//
//  Negative channel numbers are not acceptable. -1 is used to indicate an unused listener slot,
//  and other negative values may be given other special meanings.
//
//  Machines should only send during a Tick call, so that listeners do not receive commands while
//  running a Work call. Listeners should be aware that the callback can occur either before or
//  after a Tick, and ensure that both cases synchronise properly.
//
//  The problem of synchronising external callers to Tick calls is not yet settled! - Probably I'll
//  have a queue and a special call for Ninereeds Broadcast to empty the queue.
//

#ifndef __NINEREEDS_BROADCAST_LIB_HEAD
#define __NINEREEDS_BROADCAST_LIB_HEAD

#include <windows.h>

#include "../MachineInterface.h"

#ifdef NR_IMPLEMENTATION
#define DI __declspec(dllexport)
#else
#define DI __declspec(dllimport)
#endif

#define NR_MAX_LISTENERS 64

//
//  The command parameters structure
//

#pragma pack(1)

//
//  The time field is a delay from when the command is sent. Commands
//  are assumed to synchronise with Tick calls - a command is never
//  treated as if it were sent half way through a tick.
//

struct c_Broadcast_Params
{
	byte p_Command;      //  Range   01..  FF :   00 = no command
	word p_Time;         //  Range 0000..FFFE : FFFF = no value
	                     //                     High byte is whole ticks
	                     //                     Low word is fractional ticks

	byte p_Byte_Param1;  //  Range   00..  FE :   FF = no value
	byte p_Byte_Param2;
	byte p_Byte_Param3;
	byte p_Byte_Param4;

	word p_Word_Param1;  //  Range 0000..FFFE : FFFF = no value
	word p_Word_Param2;
	word p_Word_Param3;
	word p_Word_Param4;
};

#pragma pack()

//
//  Listeners will provide a callback object that inherits from the following class...
//

#define NR_RX_QUEUE_MAX 64


struct c_Control_State
{
  bool f_Override;

  int  f_Curr_Value;
  int  f_Increment;

  int  f_Error;
  int  f_Error_Increment;

  int  f_Ticks_At_Start;
  int  f_Ticks_Remaining;

  int  f_Mapped_Control;
  int  f_Mapped_Start;
  int  f_Mapped_End;
  int  f_Mapped_Diff;


  void Clear (void)
  {
    f_Override = false;
  }

  void Set (int p_Start)  //  For one shot settings
  {
    f_Override = true;

    f_Curr_Value = (((unsigned int) (p_Start * f_Mapped_Diff)) / 0xFFFE) + f_Mapped_Start;
    f_Increment  = 0;

    f_Error = 0;
    f_Error_Increment = 0;

    f_Ticks_Remaining = 0;
    f_Ticks_At_Start  = 0;
  }

  void Set (int p_Start, int p_End, int p_Ticks)  //  p_Ticks must be > 0
  {
    f_Override = true;

    f_Curr_Value = (((unsigned int) (p_Start * f_Mapped_Diff)) / 0xFFFE) + f_Mapped_Start;

    int l_Diff = (((unsigned int) (p_End * f_Mapped_Diff)) / 0xFFFE) + f_Mapped_Start - f_Curr_Value;

    f_Increment  = l_Diff / p_Ticks;

    f_Error = 0;
    f_Error_Increment = l_Diff % p_Ticks;

    f_Ticks_Remaining = p_Ticks;
    f_Ticks_At_Start  = p_Ticks;
  }

  void Update (void)
  {
    f_Curr_Value += f_Increment;
    f_Error      += f_Error_Increment;

    if (f_Error >= f_Ticks_At_Start)
    {
      f_Error -= f_Ticks_At_Start;
      f_Curr_Value++;
    }

    f_Ticks_Remaining--;
  }

  void Read (CMachineDataInput * const pi)
  {
    pi->Read (f_Mapped_Control);
    pi->Read (f_Mapped_Start);
    pi->Read (f_Mapped_End);

    f_Mapped_Diff = f_Mapped_End - f_Mapped_Start;
  }

  void Write (CMachineDataOutput * const po)
  {
    po->Write (f_Mapped_Control);
    po->Write (f_Mapped_Start);
    po->Write (f_Mapped_End);
  }

};

struct c_Control_Config
{
  int   f_ID;    //  ID sent to Set_Control function
  char *f_Name;  //  Control name shown in combo boxes on dialog
};

class c_Broadcast_Listener_CB
{
private:

  friend BOOL APIENTRY ListenDialog (HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);

	int                 f_Queue_Channel [NR_RX_QUEUE_MAX];
	c_Broadcast_Params  f_Queue_Data    [NR_RX_QUEUE_MAX];
	int                 f_Queue_Order   [NR_RX_QUEUE_MAX];

	int                 f_Queue_Size;

	CMasterInfo        *f_Master_Info;   //  Needed for samples per tick

	bool                f_Tick;

	int                 f_Transpose;
  bool                f_Ignore_Notes;
  bool                f_All_Notes_Off;

  c_Control_State     f_Controls [8];

public:

	DI  c_Broadcast_Listener_CB (void);

	//
	//  This should be called during the machine Init function...
	//

	void Set_Master_Info (CMasterInfo  *p_Master_Info)
	{
		f_Master_Info   = p_Master_Info;
	}

	//
	//  The following callbacks are intended to allow the broadcast library
	//  standard commands to work. They implement the actual manipulation
	//  of global and track parameters - something that could be awkward
	//  without a callback. Having a callback also allows the application
	//  some room for different interpretations.
	//

	virtual void Transpose_Notes  (int p_Semitones) = 0;

	virtual void All_Notes_Off    (void) = 0;
	virtual void All_Notes_Ignore (void) = 0;

  //  Control IDs run from 0 to 7, and short be mappable by the
  //  user to particular controls such as filter cutoff or whatever.
  //
  //  The p_Value range is 0x0000 to 0xFFFE. The user should be able
  //  to control the low and high values, and this control should
  //  interpolate between them.

  virtual void Set_Control           (int p_ID, int p_Value) = 0;

  //  The following will read or write control mappings in the machine
  //  specific part of a Buzz file. It is the callers responsibility to
  //  handle block identifiers etc.

  void Read_Controls (CMachineDataInput * const pi)
  {
    for (int i = 0; i < 8; i++)
    {
      f_Controls [i].Read (pi);
    }
  }

  void Write_Controls (CMachineDataOutput * const po)
  {
    for (int i = 0; i < 8; i++)
    {
      f_Controls [i].Write (po);
    }
  }

	//
	//  The following command adds an item to the queue. If the queue
	//  is full, the item is discarded.
	//

	DI void Queue_Command (int p_Channel, c_Broadcast_Params *p_Command);

	//
	//  The following functions should be called by the real Tick and Work
	//  functions in the machine.
	//
	//  The machine will still need to do other processing in Tick, but the
	//  Work function is effectively replaced by Work2.
	//

	DI void Handle_Tick (void);
	DI bool Handle_Work (float *psamples, int numsamples, int const mode);

	//
	//  The following callbacks should be implemented by the machine to
	//  handle received commands etc.
	//
	//  The one with no parameters is just the original Tick, delayed until just
	//  before the first Work2 to ensure all received commands are handled in
	//  time.
	//
	//  offset in Work2 specifies the number of samples handled in prior Work2
	//  calls for the current Work call - it allows the right place to be found
	//  in AuxBus-sourced buffers etc.

	virtual void Tick2 (void) = 0;
	virtual void Tick2 (int p_Channel, c_Broadcast_Params *p_Command) = 0;
	virtual bool Work2 (float *psamples, int numsamples, int const mode, int offset) = 0;
};

//
//  Registering and unregistering listeners
//

DI bool NR_Start_Listening (int p_Channel, c_Broadcast_Listener_CB *p_Callback);
DI void NR_Stop_Listening  (int p_Channel, c_Broadcast_Listener_CB *p_Callback);

//
//  Listening enable/channel and control mappings dialog
//

DI void NR_Dialog_Enable_Channel (c_Broadcast_Listener_CB *p_Self,
                                  bool                    *p_Enable,
                                  int                     *p_Channel,
                                  int                      p_Num_Controls,
                                  c_Control_Config        *p_Controls     );

//
//  Sending Commands
//

DI void NR_Send_Command (int p_Channel, c_Broadcast_Params      *p_Command );

#endif
